//! Field reflection utilities for MAVLink messages.
//!
//! This module defines the [`IndexedField`] struct, which represents a field within a MAVLink
//! message, and provides methods for extracting and converting field values from messages.

use mavlink_bindgen::parser::{MavField, MavMessage, MavType};
use serde::ser::SerializeStruct;
use skyward_mavlink::mavlink::Message;

use super::{MAVLINK_PROFILE, conversion::FieldLike};

/// Represents a field within a MAVLink message, including its index, parent message, and metadata.
///
/// This struct is used for reflection and dynamic field access.
#[derive(Debug, Clone)]
pub struct IndexedField {
    /// The index of the field within the message's field list.
    pub(super) id: usize,
    /// Reference to the parent MAVLink message definition.
    pub(super) msg: &'static MavMessage,
    /// Reference to the field metadata.
    pub(super) field: &'static MavField,
}

/// Macro to generate extractor methods for various types.
///
/// This macro defines a method that attempts to extract a field value from a message and cast it
/// to the specified Rust type, handling type mismatches and unsupported types.
macro_rules! extract_as_type {
    ($as_type: ty, $func: ident, $($mav_ty: ident, $rust_ty: ty),+) => {
        /// Extract the field value as the specified type.
        ///
        /// # Arguments
        /// * `message` - The MAVLink message instance.
        ///
        /// # Returns
        /// * `Ok(value)` if extraction and casting succeed.
        /// * `Err(String)` if the field is not found or type conversion fails.
        pub fn $func(&self, message: &impl Message) -> Result<$as_type, String> {
            macro_rules! downcast {
                ($value: expr, $type: ty) => {
                    Ok(*$value
                        .downcast::<$type>()
                        .map_err(|_| "Type mismatch".to_string())? as $as_type)
                };
            }

            let value = message
                .get_field(self.id)
                .ok_or("Field not found".to_string())?;
            match self.field.mavtype {
                $(MavType::$mav_ty => downcast!(value, $rust_ty),)+
                _ => Err("Field type not supported".to_string()),
            }
        }
    };
}

impl IndexedField {
    /// Returns a reference to the parent message definition.
    pub fn msg(&self) -> &MavMessage {
        self.msg
    }

    /// Returns the message ID.
    pub fn msg_id(&self) -> u32 {
        self.msg.id
    }

    /// Returns the field index within the message.
    pub fn id(&self) -> usize {
        self.id
    }

    /// Returns a reference to the field metadata.
    pub fn field(&self) -> &MavField {
        self.field
    }

    /// Returns the field name.
    pub fn name(&self) -> &str {
        &self.field.name
    }

    /// Extracts the field value as a string using debug formatting.
    ///
    /// # Arguments
    /// * `message` - The MAVLink message instance.
    ///
    /// # Returns
    /// * `Ok(String)` if the field is found.
    /// * `Err(String)` if the field is not found.
    pub fn extract_as_string<T: Message>(&self, message: &T) -> Result<String, String> {
        let value = message
            .get_field(self.id)
            .ok_or("Field not found".to_string())?;
        let str_value = format!("{:?}", value);
        Ok(str_value)
    }
}

/// ### Extractors
/// These methods allow extracting the value of a field from a message, casting
/// it to the desired type. Each method is generated by the `extract_as_type!` macro.
impl IndexedField {
    #[rustfmt::skip]
    extract_as_type!(f32, extract_as_f32,
        UInt8, u8,
        UInt16, u16,
        UInt32, u32,
        UInt64, u64,
        Int8, i8,
        Int16, i16,
        Int32, i32,
        Int64, i64,
        Float, f32,
        Double, f64
    );

    #[rustfmt::skip]
    extract_as_type!(f64, extract_as_f64,
        UInt8, u8,
        UInt16, u16,
        UInt32, u32,
        UInt64, u64,
        Int8, i8,
        Int16, i16,
        Int32, i32,
        Int64, i64,
        Float, f32,
        Double, f64
    );

    #[rustfmt::skip]
    extract_as_type!(u8, extract_as_u8,
        UInt8, u8,
        Char, char
    );

    #[rustfmt::skip]
    extract_as_type!(u16, extract_as_u16,
        UInt8, u8,
        Int8, i8,
        UInt16, u16
    );

    #[rustfmt::skip]
    extract_as_type!(u32, extract_as_u32,
        UInt8, u8,
        Int8, i8,
        UInt16, u16,
        Int16, i16,
        UInt32, u32
    );

    #[rustfmt::skip]
    extract_as_type!(u64, extract_as_u64,
        UInt8, u8,
        Int8, i8,
        UInt16, u16,
        Int16, i16,
        UInt32, u32,
        Int32, i32,
        UInt64, u64
    );

    #[rustfmt::skip]
    extract_as_type!(i8, extract_as_i8,
        Int8, i8
    );

    #[rustfmt::skip]
    extract_as_type!(i16, extract_as_i16,
        UInt8, u8,
        Int8, i8,
        Int16, i16
    );

    #[rustfmt::skip]
    extract_as_type!(i32, extract_as_i32,
        UInt8, u8,
        Int8, i8,
        UInt16, u16,
        Int16, i16,
        Int32, i32
    );

    #[rustfmt::skip]
    extract_as_type!(i64, extract_as_i64,
        UInt8, u8,
        Int8, i8,
        UInt16, u16,
        Int16, i16,
        UInt32, u32,
        Int32, i32,
        Int64, i64
    );

    #[rustfmt::skip]
    extract_as_type!(char, extract_as_char,
        UInt8, u8,
        Char, char
    );
}

impl std::hash::Hash for IndexedField {
    /// Hashes the field index and message ID for use in hash maps and sets.
    fn hash<H: std::hash::Hasher>(&self, state: &mut H) {
        self.id.hash(state);
        self.msg.id.hash(state);
    }
}

impl PartialEq for IndexedField {
    /// Compares two `IndexedField` instances for equality by field index and message ID.
    fn eq(&self, other: &Self) -> bool {
        self.id == other.id && self.msg.id == other.msg.id
    }
}

impl serde::Serialize for IndexedField {
    /// Serializes the `IndexedField` by its field index and message ID.
    fn serialize<S: serde::Serializer>(&self, serializer: S) -> Result<S::Ok, S::Error> {
        let mut state = serializer.serialize_struct("IndexedField", 3)?;
        state.serialize_field("id", &self.id)?;
        state.serialize_field("msg_id", &self.msg.id)?;
        state.end()
    }
}

impl<'de> serde::Deserialize<'de> for IndexedField {
    /// Deserializes an `IndexedField` from its field index and message ID.
    fn deserialize<D: serde::Deserializer<'de>>(deserializer: D) -> Result<Self, D::Error> {
        #[derive(serde::Deserialize)]
        struct IndexedFieldDe {
            id: usize,
            msg_id: u32,
        }

        let de = IndexedFieldDe::deserialize(deserializer)?;
        let field = de
            .id
            .to_mav_field(de.msg_id, &MAVLINK_PROFILE)
            .map_err(|u| serde::de::Error::custom(format!("Invalid field: {}", u)))?;
        Ok(field)
    }
}
