use eframe::CreationContext;
use egui::{Button, Key, Modifiers, Sides};
use egui_tiles::{Behavior, Container, Linear, LinearDir, Tile, TileId, Tiles, Tree};
use serde::{Deserialize, Serialize};
use std::{
    fs,
    ops::DerefMut,
    path::{Path, PathBuf},
    sync::{Arc, Mutex},
    time::{Duration, Instant},
};
use tracing::{debug, error, trace};

use crate::{
    error::ErrInstrument,
    mavlink::MavMessage,
    message_broker::{MessageBroker, MessageBundle},
    utils::id::PaneId,
};

use super::{
    panes::{Pane, PaneBehavior, PaneKind},
    persistency::LayoutManager,
    shortcuts::{ShortcutHandler, ShortcutMode},
    utils::maximized_pane_ui,
    widget_gallery::WidgetGallery,
    widgets::reception_led::ReceptionLed,
    windows::{ConnectionsWindow, LayoutManagerWindow},
};

pub struct App {
    /// Persistent state of the app
    state: AppState,
    layout_manager: LayoutManager,
    behavior: AppBehavior,
    maximized_pane: Option<TileId>,
    // == Message handling ==
    message_broker: MessageBroker,
    message_bundle: MessageBundle,
    // Shortcut handling
    shortcut_handler: Arc<Mutex<ShortcutHandler>>,
    // == Windows ==
    widget_gallery: WidgetGallery,
    sources_window: ConnectionsWindow,
    layout_manager_window: LayoutManagerWindow,
}

// An app must implement the `App` trait to define how the ui is built
impl eframe::App for App {
    // The update function is called each time the UI needs repainting!
    fn update(&mut self, ctx: &egui::Context, _frame: &mut eframe::Frame) {
        self.process_incoming_messages();

        // Get the id of the hovered pane, in order to apply actions to it
        let hovered_pane = self.behavior.tile_id_hovered;
        trace!("Hovered pane: {:?}", hovered_pane);

        // Capture any pane action generated by pane children
        let mut pane_action = self.behavior.action.take();
        trace!("Pane action: {:?}", pane_action);

        if let Some(hovered_tile) = hovered_pane {
            // Capture any pane action generated by keyboard shortcuts
            let key_action_pairs = [
                (Modifiers::ALT, Key::V, PaneAction::SplitV),
                (Modifiers::ALT, Key::H, PaneAction::SplitH),
                (Modifiers::ALT, Key::C, PaneAction::Close),
                (Modifiers::ALT, Key::R, PaneAction::ReplaceThroughGallery),
                (Modifiers::SHIFT, Key::Escape, PaneAction::Maximize),
                (Modifiers::NONE, Key::Escape, PaneAction::Exit),
            ];
            pane_action = pane_action.or(self
                .shortcut_handler
                .lock()
                .log_unwrap()
                .consume_if_mode_is(ShortcutMode::composition(), &key_action_pairs[..])
                .map(|a| (hovered_tile, a)));
        }

        // If an action was triggered, we consume it
        if let Some((tile_id, action)) = pane_action.take() {
            match action {
                PaneAction::SplitH => {
                    if let Some(hovered_tile) = hovered_pane {
                        profiling::scope!("split_h");
                        if self.maximized_pane.is_none() {
                            let panes_tree = &mut self.state.panes_tree;
                            debug!("Called SplitH on tile {:?}", hovered_tile);
                            let hovered_tile_pane = panes_tree
                                .tiles
                                .remove(hovered_tile)
                                .log_expect("Hovered tile not found");
                            let left_pane = panes_tree.tiles.insert_new(hovered_tile_pane);
                            let right_pane = panes_tree.tiles.insert_pane(Pane::default());
                            panes_tree.tiles.insert(
                                hovered_tile,
                                Tile::Container(Container::Linear(Linear::new_binary(
                                    LinearDir::Horizontal,
                                    [left_pane, right_pane],
                                    0.5,
                                ))),
                            );
                        }
                    }
                }
                PaneAction::SplitV => {
                    profiling::scope!("split_v");
                    if self.maximized_pane.is_none() {
                        if let Some(hovered_tile) = hovered_pane {
                            debug!("Called SplitV on tile {:?}", hovered_tile);
                            let panes_tree = &mut self.state.panes_tree;
                            let hovered_tile_pane = panes_tree
                                .tiles
                                .remove(hovered_tile)
                                .log_expect("Hovered tile not found");
                            let replaced = panes_tree.tiles.insert_new(hovered_tile_pane);
                            let lower_pane = panes_tree.tiles.insert_pane(Pane::default());
                            panes_tree.tiles.insert(
                                hovered_tile,
                                Tile::Container(Container::Linear(Linear::new_binary(
                                    LinearDir::Vertical,
                                    [replaced, lower_pane],
                                    0.5,
                                ))),
                            );
                        }
                    }
                }
                PaneAction::Close => {
                    if let Some(hovered_tile) = hovered_pane {
                        debug!("Called Close on tile {:?}", hovered_tile);
                        let panes_tree = &mut self.state.panes_tree;
                        // Ignore if the root pane is the only one
                        if panes_tree.tiles.len() != 1 && self.maximized_pane.is_none() {
                            panes_tree.remove_recursively(hovered_tile);
                        }
                    }
                }
                PaneAction::Replace(mut new_pane) => {
                    debug!(
                        "Called Replace on tile {:?} with pane {:?}",
                        tile_id, new_pane
                    );
                    new_pane.init(self.state.next_pane_id());
                    self.state
                        .panes_tree
                        .tiles
                        .insert(tile_id, Tile::Pane(*new_pane));
                }
                PaneAction::ReplaceThroughGallery => {
                    self.widget_gallery.replace_tile(tile_id);
                }
                PaneAction::Maximize => {
                    // This is a toggle: if there is not currently a maximized pane,
                    // maximize the hovered pane, otherwize remove the maximized pane.
                    if self.maximized_pane.is_some() {
                        self.maximized_pane = None;
                    } else if let Some(hovered_tile) = hovered_pane {
                        let panes_tree = &mut self.state.panes_tree;
                        let hovered_pane_is_default = panes_tree
                            .tiles
                            .get(hovered_tile)
                            .map(|hovered_pane| {
                                matches!(
                                    hovered_pane,
                                    Tile::Pane(Pane {
                                        pane: PaneKind::Default(_),
                                    })
                                )
                            })
                            .unwrap_or(false);
                        if !hovered_pane_is_default {
                            self.maximized_pane = Some(hovered_tile);
                        }
                    }
                }
                PaneAction::Exit => {
                    if self.maximized_pane.is_some() {
                        self.maximized_pane = None;
                    }
                }
            }
        }

        // TODO: maybe introduce a stats struct to store these values?
        let reception_led_active = self
            .message_broker
            .time_since_last_reception()
            .unwrap_or(Duration::MAX)
            < Duration::from_millis(100);
        let reception_frequency = self.message_broker.reception_frequency();

        // Show a panel at the bottom of the screen with few global controls
        egui::TopBottomPanel::bottom("bottom_control").show(ctx, |ui| {
            // Horizontal belt of controls
            Sides::new().show(
                ui,
                |ui| ui.add(ReceptionLed::new(reception_led_active, reception_frequency)),
                |ui| {
                    ui.horizontal(|ui| {
                        egui::global_theme_preference_switch(ui);

                        // Window for the sources
                        self.sources_window
                            .show_window(ui, &mut self.message_broker);

                        if ui
                            .add(Button::new("🔌").frame(false))
                            .on_hover_text("Open the Sources")
                            .clicked()
                        {
                            self.sources_window.visible = !self.sources_window.visible;
                        }
                        if ui
                            .add(Button::new("💾").frame(false))
                            .on_hover_text("Open the Layout Manager")
                            .clicked()
                        {
                            self.layout_manager_window
                                .toggle_open_state(&self.layout_manager);
                        }

                        // If a pane is maximized show a visual clue
                        if self.maximized_pane.is_some() {
                            ui.label("Pane Maximized!");
                        }
                    });
                },
            );
        });

        // A central panel covers the remainder of the screen, i.e. whatever area is left after adding other panels.
        egui::CentralPanel::default().show(ctx, |ui| {
            let panes_tree = &mut self.state.panes_tree;
            if let Some(maximized_pane) = self.maximized_pane {
                if let Some(Tile::Pane(pane)) = panes_tree.tiles.get_mut(maximized_pane) {
                    maximized_pane_ui(
                        ui,
                        pane,
                        self.shortcut_handler.lock().log_unwrap().deref_mut(),
                    );
                } else {
                    unreachable!("Maximized pane not found in tree!");
                }
            } else {
                panes_tree.ui(&mut self.behavior, ui);
            }
        });

        self.layout_manager_window
            .show(ctx, &mut self.layout_manager, &mut self.state);
        if let Some(action) = self.widget_gallery.show(ctx) {
            debug!("Widget gallery returned action {action:?}");
            self.behavior.action = Some(action);
        }

        // Process outgoing messages
        self.process_outgoing_messages();

        // Used for the profiler
        profiling::finish_frame!();

        // UNCOMMENT THIS TO ENABLE CONTINOUS MODE
        // ctx.request_repaint();
    }

    fn save(&mut self, storage: &mut dyn eframe::Storage) {
        self.layout_manager.save_current_layout(storage);
    }
}

impl App {
    pub fn new(ctx: &CreationContext) -> Self {
        // Load the image loaders
        egui_extras::install_image_loaders(&ctx.egui_ctx);

        // Install the fonts
        super::font::add_font(&ctx.egui_ctx);

        let mut layout_manager = if let Some(storage) = ctx.storage {
            LayoutManager::new(storage)
        } else {
            LayoutManager::default()
        };

        let mut state = AppState::default();

        // Load the selected layout if valid and existing
        if let Some(layout) = layout_manager.current_layout().cloned() {
            layout_manager
                .load_layout(layout, &mut state)
                .unwrap_or_else(|e| {
                    error!("Error loading layout: {}", e);
                });
        }

        let shortcut_handler = Arc::new(Mutex::new(ShortcutHandler::new(ctx.egui_ctx.clone())));
        Self {
            state,
            layout_manager,
            message_broker: MessageBroker::new(ctx.egui_ctx.clone()),
            widget_gallery: WidgetGallery::default(),
            behavior: AppBehavior::new(Arc::clone(&shortcut_handler)),
            maximized_pane: None,
            message_bundle: MessageBundle::default(),
            shortcut_handler,
            sources_window: ConnectionsWindow::default(),
            layout_manager_window: LayoutManagerWindow::default(),
        }
    }

    /// Retrieves new messages from the message broker and dispatches them to the panes.
    #[profiling::function]
    fn process_incoming_messages(&mut self) {
        let start = Instant::now();

        self.message_broker
            .process_incoming_messages(&mut self.message_bundle);

        // Skip updating the panes if there are no messages
        let count = self.message_bundle.count();
        if count == 0 {
            return;
        }

        debug!(
            "Receiving {count} messages from message broker took {:?}",
            start.elapsed()
        );

        let start = Instant::now();
        for (_, tile) in self.state.panes_tree.tiles.iter_mut() {
            // Skip non-pane tiles
            let Tile::Pane(pane) = tile else { continue };
            // Skip panes that do not have a subscription
            let sub_ids: Vec<u32> = pane.get_message_subscriptions().collect();

            if pane.should_send_message_history() {
                pane.update(self.message_broker.get(&sub_ids[..]).as_slice());
            } else {
                pane.update(self.message_bundle.get(&sub_ids[..]).as_slice());
            }
        }

        debug!(
            "Panes message processing messages took {:?}",
            start.elapsed()
        );
        self.message_bundle.reset();
    }

    /// Sends outgoing messages from the panes to the message broker.
    #[profiling::function]
    fn process_outgoing_messages(&mut self) {
        let outgoing: Vec<MavMessage> = self
            .state
            .panes_tree
            .tiles
            .iter_mut()
            .filter_map(|(_, tile)| {
                if let Tile::Pane(pane) = tile {
                    Some(pane.drain_outgoing_messages())
                } else {
                    None
                }
            })
            .flatten()
            .collect();
        self.message_broker.process_outgoing_messages(outgoing);
    }
}

#[derive(Serialize, Deserialize, Clone, PartialEq)]
pub struct AppState {
    pub panes_tree: Tree<Pane>,
    pub next_pane_id: PaneId,
}

impl Default for AppState {
    fn default() -> Self {
        let mut tiles = Tiles::default();
        let root = tiles.insert_pane(Pane::default());
        let panes_tree = egui_tiles::Tree::new("main_tree", root, tiles);
        let next_pane_id = PaneId::new(0);

        Self {
            panes_tree,
            next_pane_id,
        }
    }
}

impl AppState {
    pub fn from_file(path: &PathBuf) -> anyhow::Result<Self> {
        fs::read_to_string(path)
            .and_then(|json| serde_json::from_str::<AppState>(&json).map_err(Into::into))
            .map_err(|e| anyhow::anyhow!("Error deserializing layout: {}", e))
    }

    pub fn to_file(&self, path: &Path) -> anyhow::Result<()> {
        // Check if the parent path exists, if not create it
        if let Some(parent) = path.parent() {
            if !parent.exists() {
                fs::create_dir_all(parent)
                    .map_err(|e| anyhow::anyhow!("Error creating directory: {}", e))?;
                debug!("Created directory {:?}", parent);
            }
        }

        let serialized_layout = serde_json::to_string_pretty(self)
            .map_err(|e| anyhow::anyhow!("Error serializing layout: {}", e))?;
        debug!("Serialized layout: {}", serialized_layout);
        fs::write(path, serialized_layout)
            .map_err(|e| anyhow::anyhow!("Error writing layout: {}", e))?;

        Ok(())
    }

    pub fn next_pane_id(&mut self) -> PaneId {
        let id = self.next_pane_id;
        self.next_pane_id = self.next_pane_id.next_id();
        id
    }
}

/// Behavior for the tree of panes in the app
pub struct AppBehavior {
    pub shortcut_handler: Arc<Mutex<ShortcutHandler>>,
    pub action: Option<(TileId, PaneAction)>,
    pub tile_id_hovered: Option<TileId>,
}

impl AppBehavior {
    fn new(shortcut_handler: Arc<Mutex<ShortcutHandler>>) -> Self {
        Self {
            shortcut_handler,
            action: None,
            tile_id_hovered: None,
        }
    }
}

impl Behavior<Pane> for AppBehavior {
    fn pane_ui(
        &mut self,
        ui: &mut egui::Ui,
        tile_id: TileId,
        pane: &mut Pane,
    ) -> egui_tiles::UiResponse {
        let res = ui.scope(|ui| pane.ui(ui, self.shortcut_handler.lock().log_unwrap().deref_mut()));
        let PaneResponse {
            action_called,
            drag_response,
        } = res.inner;

        // Check if the pointer is hovering over the pane
        if res.response.contains_pointer() {
            self.tile_id_hovered = Some(tile_id);
        }

        // Capture the action and store it to be consumed in the update function
        if let Some(action_called) = action_called {
            self.action = Some((tile_id, action_called));
        }
        drag_response
    }

    fn tab_title_for_pane(&mut self, _pane: &Pane) -> egui::WidgetText {
        "Tab".into()
    }
}

#[derive(Clone)]
pub struct PaneResponse {
    pub action_called: Option<PaneAction>,
    pub drag_response: egui_tiles::UiResponse,
}

impl PaneResponse {
    pub fn set_action(&mut self, action: PaneAction) {
        self.action_called = Some(action);
    }

    pub fn set_drag_started(&mut self) {
        self.drag_response = egui_tiles::UiResponse::DragStarted;
    }
}

impl Default for PaneResponse {
    fn default() -> Self {
        Self {
            action_called: None,
            drag_response: egui_tiles::UiResponse::None,
        }
    }
}

#[derive(Clone, Debug)]
pub enum PaneAction {
    SplitH,
    SplitV,
    Close,
    Replace(Box<Pane>),
    ReplaceThroughGallery,
    Maximize,
    Exit,
}
