function [value, isterminal, direction] = eventAB(t, Y, t0, iAB, settings, varargin)
%{
eventAB - Event function to stop simulation when the AB configuration
          change

INPUTS:
        - t, double [n° variations, 1], integration time, [s];
        - Y, double [n° variations, 13], state matrix,
                            [ x y z | u v w | p q r | q0 q1 q2 q3]:
                            * (x y z), NED{north, east, down} horizontal frame;
                            * (u v w), body frame velocities;
                            * (p q r), body frame angular rates;
                            * (q0 q1 q2 q3), attitude unit quaternion.
        - t0, double [1, 1], previous AB configration change time;
        - iAB, double [1, 1], i-th airbrake configuration;
        - settings, struct, rocket and simulation data.

OUTPUTS:
        - value, selected value to check if the integration has to be stopped (vertical velocity);
        - isterminal, logical input to stop the integration;
        - direction, to select the sign that the function must have when tends to zero, 1 = positive.

CALLED FUNCTIONS: -

REVISIONS:
- 0    02/10/2021, Release, Davide Rosato
Copyright © 2021, Skyward Experimental Rocketry, AFD department
All rights reserved

SPDX-License-Identifier: GPL-3.0-or-later

%}

if iAB == 1
    % recalling the states
    z = Y(3);
    u = Y(4);
    v = Y(5);
    w = Y(6);
    q0 = Y(10);
    q1 = Y(11);
    q2 = Y(12);
    q3 = Y(13);

    %% CONSTANTS
    local = settings.Local;                 % vector containing inputs for atmosphereData


    g = settings.g0/(1 + (-z*1e-3/6371))^2; % [N/kg]  module of gravitational field

    if settings.stoch.N > 1
        uncert = settings.stoch.uncert;
        Day = settings.stoch.Day;
        Hour = settings.stoch.Hour;
        uw = settings.stoch.uw; vw = settings.stoch.vw; ww = settings.stoch.ww;
    else
        uncert = [0, 0];

        if not(settings.wind.input) && not(settings.wind.model)
            uw = settings.constWind(1); vw = settings.constWind(2); ww = settings.constWind(3);
        end
    end

    %% QUATERION ATTITUDE
    Q = [q0 q1 q2 q3];
    Q = Q/norm(Q);

    %% ADDING WIND (supposed to be added in NED axes);
    if settings.wind.model

        if settings.stoch.N > 1
            [uw, vw, ww] = windMatlabGenerator(settings, z, t, Hour, Day);
        else
            [uw, vw, ww] = windMatlabGenerator(settings, z, t);
        end

    elseif settings.wind.input
        [uw, vw, ww] = windInputGenerator(settings, z, uncert);
    end

    dcm = quatToDcm(Q);
    wind = dcm*[uw; vw; ww];

    % Relative velocities (plus wind);
    ur = u - wind(1);
    vr = v - wind(2);
    wr = w - wind(3);

    V_norm = norm([ur vr wr]);

    %% ATMOSPHERE DATA
    if -z < 0     % z is directed as the gravity vector
        z = 0;
    end

    absoluteAltitude = -z + settings.z0;
    [~, a, ~, ~] = atmosphereData(absoluteAltitude, g, local);

    M = V_norm/a;

    %% Control sequence
    if t >= settings.tControl && M <= settings.MachControl
        value = 0;
        isterminal = 1;
        direction = 0;
    else
        value = 1;
        isterminal = 1;
        direction = 0;
    end

elseif iAB == length(settings.control)
    Q = Y(10:13)';

    % Inertial Frame velocities
    vels = quatrotate(quatconj(Q),Y(4:6)');

    % Stop checking if I'm in Propulsion Phase
    if t > settings.tb
        value = vels(3);
    else
        value = 1;
    end

    isterminal = 1;
    direction = 1;

elseif iAB ~= 1 && iAB ~= length(settings.control)
    value = t0 + settings.dtControl(iAB - 1) - t;
    isterminal = 1;
    direction = 0;
end


end
