function [T, a, P, rho] = atmosphereData(absoluteAltitude, g, local)

%{
atmosphereData - This function computes atmosphere data at a given
altitude, based on the ISA model, adjusted with real data at a local
altitude

INPUTS:
        - absoluteAltitude: absolute were atmosphere data need to be
                            computed [m];
        - g: gravitational costant [kg/m^2];
        - local: (1x4) [altLocal, TLocal, PLocal, rhoLocal]

OUTPUT:
        - T: temperature [K];
        - a: speed of sound [m/s];
        - P: pressure [Pa];
        - rho: density [kg/m^3];
REVISIONS:
- 1     18/10/21,   update     Jacopo Carradori, Simone Cimolato, Riccardo
                               Cadamuro: adjustment with real data at a
                               given altitude added.

Copyright © 2021, Skyward Experimental Rocketry, AFD department
All rights reserved

SPDX-License-Identifier: GPL-3.0-or-later

%}

atmCostants = [0.0065, 401.87434, 1.86584515, 1.225, ...
               101325, 288.15, 287.05];           % atmosisa constants: [a, gamma*R, R*a, rho0, p0, T0, R]

rho0 = atmCostants(4);
P0 = atmCostants(5);
T0 = atmCostants(6);

if length(local) == 4                             % if T, P, rho (at local altitude) and local altitude are given as inputs, T0, P0 and rho0 are computed in order to have a more accurate atmosphere model
    altLocal = local(1);
    TLocal = local(2);
    PLocal = local(3);
    rhoLocal = local(4);
    T0 = TLocal + atmCostants(1)*altLocal;
    theta0 = TLocal/T0;
    P0 = PLocal/(theta0^(g/atmCostants(3)));
    rho0 = rhoLocal/(theta0^(g/atmCostants(3)-1.0));
elseif length(local) == 3                         % if T, P (at local altitude) and local altitude are given as inputs, T0, P0 and rho0 are computed in order to have a more accurate atmosphere model
    altLocal = local(1);
    TLocal = local(2);
    PLocal = local(3);
    T0 = TLocal + atmCostants(1)*altLocal;
    theta0 = TLocal/T0;
    P0 = PLocal/(theta0^(g/atmCostants(3)));
    rho0 = P0/(atmCostants(7)*T0);                % based on the hypothesis that air is an ideal gas
elseif length(local) == 2                         % if T (at a local altitude) and local altitude are given as inputs, T0, P0 and rho0 are computed in order to have a more accurate atmosphere model
    altLocal = local(1);
    TLocal = local(2);
    T0 = TLocal + atmCostants(1)*altLocal;
    rho0 = P0/(atmCostants(7)*T0);                % based on the hypothesis that air is an ideal gas
end
T = T0 - atmCostants(1)*absoluteAltitude;
a = sqrt(T*atmCostants(2));
theta = T/T0;
P = P0*theta.^(g/atmCostants(3));
rho = rho0*theta.^((g/atmCostants(3)) - 1.0);
end