function Vq = interpLinear(X, V, Xq, allowExtrapolation)
arguments
    X                   double
    V                   double
    Xq                  double
    allowExtrapolation  logical = false
end
%{
This function interpolates linearly.
It works only with a scalar as point to interpolate (third input),

INPUTS:
        - X, vector of sample points   (x dei nodi)
        - V, vector of values of sample points   (y dei nodi)
        - Xq, query point    (x segnata)

OUTPUT:
        - Vq, value of the query point   (y segnata)

VERSION:
        - 04/11/2021,   release,    Jacopo Carradori
        - 15/08/2022,   update,     Davide Rosato
                        More dimension added.
        - 28/10/2023,   update,     Marco Luigi Gaibotti
                        Vectorized range selection

Copyright © 2022, Skyward Experimental Rocketry, AFD department
All rights reserved

SPDX-License-Identifier: GPL-3.0-or-later

%}

Lx = size(X, 2);

if Lx ~= size(V, 2), error ('Length of X and V should be the same'); end

if ~allowExtrapolation
    if Xq > X(end) || Xq < X(1)
        error('It is an extrapolation, requesteed value is out of bound'); 
    end
else
    if Xq >= X(end)
        Vq = V(end);
        return;
    elseif Xq <= X(1)
        Vq = V(1);
        return;
    end
end

if Xq == X(1), Vq = V(:, 1); return; end

Xu = Xq>X;
i = sum(Xu);
if ~all(Xu(1:i)), error("X vector not suitable for interpolation, check for duplicates"); end
Vq = (V(:,i+1)-V(:,i))/(X(i+1)-X(i))*(Xq-X(i)) + V(:,i);
end