function [val, err] = interpN(MAT, inV, inQ)
%{

linear interpolation in N dimensions

INPUTS:
      MAT: double [N-dimensional] matrix containing the coefficients to
              interpolate, the first dimension regards the coefficient number,
              the others regards the dependencies
      inV: cell {1, N-1} each cell contains the vector referred
              to the (i+1)th dimension in MAT 
      inQ: double [1, N] vector containing the points in wich interpolate
  
OUTPUTS:
      val: double [1,1] result of the interpolation
      err: double [1,1] error flag:
                        -0 no error
                        -1 extrapolation performed

REVISIONS:
- 0     01/12/2022, release     Riccardo Cadamuro
- 1     17/02/2023, update      Riccardo Cadamuro
                    minor improvement

Copyright © 2022, Skyward Experimental Rocketry, AFD department
All rights reserved

SPDX-License-Identifier: GPL-3.0-or-later

%}

    nV = length(inV); 
    dim0 = size(MAT);
    err = false; 

    for i = 1:nV
        vec = inV{i}; 
        nEls = length(vec); 
        ind1 = nEls-1; 
        ind2 = nEls;
        
        flag = false; 
        for j = 1:nEls
            if  inQ(i) <= vec(j)
                
                if j == 1
                    ind1 = j; 
                    ind2 = j + 1;
                else
                    ind1 = j - 1;
                    ind2 = j; 
                    flag = true;    % NON-extrapolation
                end
                break; 
            end
        end
        
        if err == false             
            % if extrapolation along one dimenion only, the final risult is considerd faulty
            err = (~flag); 
        end

        x1 = vec(ind1); 
        x2 = vec(ind2); 
       
        dimVec = [dim0(1), dim0(i+2:end)];
        if i == nV 
            dimVec(2) = 1; 
        end

        y1 = MAT(:, ind1, :); 
        y2 = MAT(:, ind2, :); 

    
        MAT = (inQ(i) - x1)./(x2 - x1) .* (y2 - y1) + y1; 
        MAT = reshape(MAT, dimVec); 

    end
    
    val = MAT; 
end