function [dY, parout] = descentParachute(~, Y, rocket, environment, wind, settings, descentData, wrapper)
arguments
    ~
    Y
    rocket          Rocket
    environment     Environment
    wind            % WindCustom {mustBeA(wind, {'WindCustom', 'WindMatlab'})}
    settings        Settings
    descentData     struct
    wrapper                = [] %DataWrapper = DataWrapper.empty
end
% descentParachute - ode function of the 3DOF parachute model
% 
% INPUTS:
% - t,       double [1, 1] integration time [s];
% - Y,       double [6, 1] state vector [ x y z | u v w | p q r | q0 q1 q2 q3 | angle]:
%                                 * (x y z), NED{north, east, down} horizontal frame;
%                                 * (u v w), body frame velocities;
%
% - (rocket, environment, wind), data Classes
% - settings, struct
% - descentData, struct
%       * para, double,  current parachute (para matrix row)
%       * stage, double, current stage     (para matrix col)
% - wrapper, handle to export data
% 
% OUTPUTS:
% - dY,      double [6, 1] state derivatives
% - parout,  struct, additional flight qunatities
%
% Copyright © 2021, Skyward Experimental Rocketry, AFD, GNC departments
% All rights reserved
% 
% SPDX-License-Identifier: GPL-3.0-or-later

%% recalling the state
% x = Y(1);
% y = Y(2);
altitude = -Y(3);
u = Y(4);
v = Y(5);
w = Y(6);

dY = zeros(6, 1);

%% GETTING DATA          

if isfield(settings.simulator, 'stochNumber')
    para = settings.stoch.para; %!!!!!!!!!!
else
    para = descentData.para; % defined in stdRun {para = i; settings.paraNumber = para;}
end
stage = descentData.stage;

S = rocket.parachutes(para, stage).surface;          % [m^2]   Surface
CD = rocket.parachutes(para, stage).cd;        % [/] Parachute Drag Coefficient
CL = rocket.parachutes(para, stage).cl;        % [/] Parachute Lift Coefficient

m = rocket.stagesMass(stage);

g = environment.g0/(1 + (altitude*1e-3/6371))^2;                                         % [N/kg]  module of gravitational field
local = [environment.z0, environment.temperature, ...   % vector containing inputs for atmosphereData
    environment.pressure, environment.rho];

%% ADDING WIND (supposed to be added in NED axes);
switch class(wind)
    case 'WindMatlab'
        [uw, vw, ww] = wind.getVels(altitude, theta);
    case 'WindCustom'
        [uw, vw, ww] = wind.getVels(altitude);
end

windVels = [uw vw ww];

% Relative velocities (plus wind);
ur = u - windVels(1);
vr = v - windVels(2);
wr = w - windVels(3);

V_norm = norm([ur vr wr]);

%% ATMOSPHERE DATA
absoluteAltitude = altitude + environment.z0;
[~, ~, P, rho] = atmosphereData(absoluteAltitude, g, local);

%% REFERENCE FRAME
% The parachutes are approximated as rectangular surfaces with the normal
% vector perpendicular to the relative velocity
t_vect = [ur vr wr];                     % Tangenzial vector
h_vect = [vr -ur 0];                     % horizontal vector    

if all(abs(h_vect) < 1e-8)
    h_vect = [vw -uw 0];
end

n_vect = cross(t_vect, h_vect);          % Normal vector

if norm(t_vect) < 1e-8
    t_vers = [0 0 0];
else
    t_vers = t_vect/norm(t_vect);            % Tangenzial versor
end

if norm(n_vect) < 1e-8
    n_vers = [0 0 0]; 
else
    n_vers = n_vect/norm(n_vect);           % normal versor
end

if (n_vers(3) > 0)                          % If the normal vector is downward directed
    n_vers = -n_vers;
end

%% FORCES
D = -0.5*rho*V_norm^2*S*CD*t_vers';       % [N] Drag vector
L = 0.5*rho*V_norm^2*S*CL*n_vers';       % [N] Lift vector
Fg = m*g*[0 0 1]';                       % [N] Gravitational Force vector
F = L + Fg + D;                          % [N] total forces vector

%% STATE DERIVATIVES
% velocity
du = F(1)/m;
dv = F(2)/m;
dw = F(3)/m;

%% FINAL DERIVATIVE STATE ASSEMBLING
dY(1:3) = [u; v; w];
dY(4) = du;
dY(5) = dv;
dY(6) = dw;

%% SAVING THE QUANTITIES FOR THE PLOTS
if nargout == 2 || ~isempty(wrapper)
    parout.state = [];

    parout.interp.alt = altitude;
    parout.interp.mass = m;

    parout.wind.NED = [uw; vw; ww];

    parout.rotations = [];

    parout.velocities = [u; v; w];

    parout.forces.aero = [];

    parout.air.rho = rho;
    parout.air.P = P;

    parout.accelerations.body = [du; dv; dw];
    
    parout.coeff = [];
    parout.uncertanty = [];

    if ~isempty(wrapper)
        wrapper.setCache(parout);
    end
end