function [dY,parout] = descentParafoil(t, Y, rocket, environment, wind, ~, descentData, wrapper) 
arguments
    t
    Y
    rocket      Rocket
    environment Environment
    wind        % WindCustom {mustBeA(wind, {'WindCustom', 'WindMatlab'})}
    ~
    descentData struct
    wrapper                = [] %DataWrapper = DataWrapper.empty
end
% ballistic - ode function of the 6DOF Rigid Rocket Model
% 
% INPUTS:
% - t,       double [1, 1] integration time [s];
% - Y,       double [13, 1] state vector [ x y z | u v w | p q r | q0 q1 q2 q3 | angle]:
%                                 * (x y z), NED{north, east, down} horizontal frame;
%                                 * (u v w), body frame velocities;
%                                 * (p q r), body frame angular rates;
%                                 * (q0 q1 q2 q3), attitude unit quaternion;
%
% - (rocket, environment, wind), data Classes
% - wrapper, handle to export data
% 
% OUTPUTS:
% - dY,      double [13, 1] state derivatives
% - parout,  struct, additional flight qunatities
%
% Copyright © 2021, Skyward Experimental Rocketry, AFD, GNC departments
% All rights reserved
% 
% SPDX-License-Identifier: GPL-3.0-or-later

%% recall states
% x = Y(1);
% y = Y(2);
altitude = -Y(3);
u = Y(4);
v = Y(5);
w = Y(6);
p = Y(7);
q = Y(8);
r = Y(9);
qw = Y(10); % scalar first
qx = Y(11);
qy = Y(12);
qz = Y(13);
deltaA = 0;

dY = zeros(13, 1);
parafoil = rocket.parachutes(descentData.para, descentData.stage);

% saturation on servo angle, needed if the actuation is faster than the
% integration step
% if deltaA > contSettings.parafoil.uMax
%     deltaA = contSettings.parafoil.uMax;
%     flagAngleSaturation = true;
% elseif deltaA < contSettings.parafoil.uMin
%     deltaA = contSettings.parafoil.uMin;
%     flagAngleSaturation = true;
% else
%     flagAngleSaturation = false;
% end

%% CONSTANTS
% environment
g = environment.g0/(1 + (altitude*1e-3/6371))^2; % [N/kg]  module of gravitational field
local = environment.local;                       % vector containing inputs for atmosphereData

% geometry
semiWingSpan = parafoil.semiWingSpan;                         % [m] wingspan
MAC = parafoil.MAC;                                           % [m] mean aero chord
surface = parafoil.surface;                                   % [m^2] payload surface
mass = parafoil.mass;                                         % [kg]
inertia = parafoil.inertia;                                   % 3x3 inertia matrix
inverseInertia = parafoil.inverseInertia;                     % 3x3 inverse inertia matrix

% aerodynamic coefficients
cd0 = parafoil.cd0; cdAlpha = parafoil.cdAlpha;
cl0 = parafoil.cl0; clAlpha = parafoil.clAlpha;
cM0 = parafoil.cM0; cMAlpha = parafoil.cMAlpha; cMQ = parafoil.cMQ;
cNR = parafoil.cNR;
cLP = parafoil.cLP; ClPhi = parafoil.cLPhi;

% aerodynamic control coefficients - asymmetric
cNSurface = parafoil.cNSurface;
cdSurface = parafoil.cdSurface;
cLSurface = parafoil.cLSurface;
clSurface = parafoil.clSurface;

% aerodynamic control coefficients - symmetric
deltaSMax = parafoil.deltaSMax; % max value


%% ROTATIONS
Q = [qw qx qy qz]; % we want it scalar first
Q = Q/norm(Q);

%% ADDING WIND (supposed to be added in NED axes);
if isa(wind, 'WindMatlab')
    [uw, vw, ww] = wind.getVels(altitude, t);
else
    [uw, vw, ww] = wind.getVels(altitude);
end

% rotation ned to body
dcm = quatToDcm(Q); % we want it scalar first
eul = quat2eul(Q);  % we want it scalar first, output PSI, THETA, PHI
eul = flip(eul,2);    % to have PHI, THETA, PSI
windVels = dcm*[uw; vw; ww];

% Relative velocities
ur = abs(u - windVels(1)); % abs to evade problem of negative vx body in forces computation
vr = v - windVels(2);
wr = w - windVels(3);

% Body to Inertial velocities
vNED = dcm'*[u; v; w];

% relative velocity norm
nNorm = norm([ur vr wr]);

%% ATMOSPHERE DATA
absoluteAltitude = altitude + environment.z0;
[~, ~, ~, rho] = atmosphereData(absoluteAltitude, g, local);

%% AERODYNAMICS ANGLES
if not(abs(ur) < 1e-9 || nNorm < 1e-9)
    alpha = atan(wr/ur);
    % beta = atan(vr/ur);                         % beta = asin(vr/V_norm) is the classical notation, Datcom uses this one though.
    % alpha_tot = atan(sqrt(wr^2 + vr^2)/ur);   % datcom 97' definition
else
    alpha = 0;
    % beta = 0;
end

%% controls
deltaANormalized = deltaA / deltaSMax;

%% FORCES
qFactor = 0.5*rho*nNorm;            % [Pa * s / m] dynamic pressure/vNorm

lift = qFactor * (cl0 + alpha * clAlpha + deltaANormalized * cLSurface) * [wr; 0; -ur];
drag = qFactor * (cd0 + alpha^2 * cdAlpha + deltaANormalized * cdSurface) * [ur; vr; wr];

forceAero = (lift - drag)*surface;

% Inertial to body gravity force (in body frame):
Fg = dcm*[0; 0; mass*g];        % [N] force due to the gravity in body frame

% total force assembly
F = Fg + forceAero;             % [N] total forces vector

%% MOMENT
Mx = (eul(1) * ClPhi *nNorm + 0.5 * semiWingSpan* cLP * p + deltaANormalized * clSurface *nNorm) * semiWingSpan;
My = (alpha * cMAlpha *nNorm+ 0.5 * MAC * cMQ * q + cM0*nNorm) * MAC;
Mz = (r * 0.5 * semiWingSpan* cNR + deltaANormalized * cNSurface*nNorm) * semiWingSpan; % bizzarre

momentAero = [Mx; My; Mz] * qFactor  * surface;
%% derivatives computations
omega = [p; q; r];

% acceleration
bodyAcc = F/mass - cross(omega,[u;v;w]);

%% angular velocity - as msa does
OM = [ 0 -p -q -r  ;
    p  0  r -q  ;
    q -r  0  p  ;
    r  q -p  0 ];

dQQ = 1/2*OM*Q';

%% angular acceleration
angAcc = inverseInertia*(momentAero - cross(omega, inertia * omega));

% %% actuator dynamics
%
% % set velocity of servo (controller)
% if ~settings.identification
%     if length(deltaA_ref_vec)==2 % for the recallOdeFunction
%         if t < t_change_ref
%             deltaA_ref = deltaA_ref_vec(1);
%         else
%             deltaA_ref = deltaA_ref_vec(2);
%         end
%     else
%         [~,ind_deltaA] = min(settings.parout.partial_time-t);
%         deltaA_ref = deltaA_ref_vec(ind_deltaA); % don't delete this unless you change how the recall ode works.
%     end
% else
%     [~,idx_deltaA] = min(abs(t-deltaA_ref_vec(:,1))); % if we are trying to identify we need to have the same input of the flight
%     deltaA_ref = deltaA_ref_vec(idx_deltaA,2);
% end
%
% ddeltaA = (deltaA_ref-deltaA)/contSettings.parafoil.deltaA_tau;
% if abs(ddeltaA) >contSettings.parafoil.deltaA_maxSpeed
%     ddeltaA = sign(deltaA_ref-deltaA)*contSettings.parafoil.deltaA_maxSpeed; % concettualmente sta roba è sbagliata perchè dipende dal passo di integrazione, fixare
% end
%
% if flagAngleSaturation
%     ddeltaA = 0;
% end

%% FINAL DERIVATIVE STATE ASSEMBLING
dY(1:3) = vNED;
dY(4:6) = bodyAcc;
dY(7:9) = angAcc;
dY(10:13) = dQQ;

if nargout == 2 || ~isempty(wrapper)
    parout.state = [];
    parout.interp = [];

    parout.wind.body = windVels;
    parout.wind.NED = [uw; vw; ww];

    parout.rotations = [];

    parout.velocities = [u; v; w];

    parout.forces.aero = forceAero;
    parout.forces.gravity = Fg;

    parout.air = [];

    parout.accelerations.body = bodyAcc;
    
    parout.coeff = [];
    parout.uncertanty = [];

    if ~isempty(wrapper)
        wrapper.setCache(parout);
    end
end