function dY = launchPadFreeDyn(t, Y, rocket, environment, settings, newSettings, Q0, CA)
arguments
    t
    Y
    rocket          Rocket
    environment     Environment
    settings        struct
    newSettings     Settings
    Q0              double
    CA              double
end
%{
launchPadFreeDyn - ODE-Function to compute the dynamics on the launchpad

INPUTS:
- t,        double [1, 1], integration time  [s];
- Y,        double [4, 1], integration state, check launchPadFreeDyn for explanation; 
                           Y = ( x y z | u )
                           (x y z): NED Earth's Surface Centered Frame ("Inertial") coordinates
                            u: body frame velocity
- settings, struct (motor, CoeffsE, CoeffsF, para, ode, stoch, prob, wind), 
                   simulation data;
- Q0,       double [4, 1], quaternion attitude [/];
- CA,       double [1, 1], axial coefficient, supposed costant [/]

OUTPUTS:
- dY,       double [4, 1], state derivatives;

CALLED FUNCTIONS: /

REVISIONS:
- 0     21/10/20,   release     Adriano Filippo Inno
Copyright © 2021, Skyward Experimental Rocketry, AFD department
All rights reserved

SPDX-License-Identifier: GPL-3.0-or-later

%}

%% STATES RECALL
% x = Y(1);
% y = Y(2);
altitude = -Y(3);
u = Y(4);

%% QUATERION ATTITUDE
conjQ = [Q0(1) -Q0(2:4)'];

% Body to Inertial velocities
Vels = quatrotate(conjQ, [u 0 0]);

%% CONSTANTS
S = rocket.crossSection;      % [m^2] cross surface
g = environment.g0;     % [N/kg] module of gravitational field at zero

absoluteAltitude = altitude + environment.z0;
local = [environment.z0, environment.temperature, ...   % vector containing inputs for atmosphereData
    environment.pressure, environment.rho];

[~, ~, ~, rho] = atmosphereData(absoluteAltitude, g, local);

omega = environment.omega;
m = interpLinear(rocket.motor.time, rocket.mass, t);
T = interpLinear(rocket.motor.time, rocket.motor.thrust, t);
% m = settings.ms + interpLinear(settings.motor.expTime, settings.motor.expM, t);

%% Dynamics
Fg = m*g*sin(omega); % [N] force due to the gravity
X = 0.5*rho*u^2*S*CA;
du = (-Fg +T -X)/m;

if T < Fg            % No velocity untill T = Fg
    du = 0;
end

%% FINAL DERIVATIVE STATE ASSEMBLING
dY(1:3) = Vels;
dY(4) = du;
dY = dY';
end