function windMat = stochWindData(settings, N)
%{
stochWindData - This function sets the wind conditions for the stochastic simulations
In particular the size of windMat will depend on the wind model chosen; 

- wind input > windMat(:, 1), uncert(1)
             > windMat(:, 2), uncert(2)

- wind variable > windMat(:, 1), wind magnitude ground (magG)
                > windMat(:, 2), wind azimuth ground (azG)
                > windMat(:, 3), limit altitude ground (hG)
                > windMat(:, 4), wind magnitude high-altitude (magH)
                > windMat(:, 5), wind azimuth high-altitude (azH)
                > windMat(:, 6), limit altitude high-altitude (hH)

- wind model > windMat(:, 1), Day
               windMat(:, 2), Hour

- wind constant > windMat(:, 1), uw
                > windMat(:, 2), vw
                > windMat(:, 3), ww
                > windMat(:, 4), Azw

INPUTS:     
            - settings, struct, stores data of the rocket and of the simulation.
            - N, double, number of stochastic simulations

OUTPUTS:
            - windMat, double [N, k]: complete wind data matrix for the whole set of N 
                                      simulations

CALLED FUNCTIONS: 

REVISIONS:
- 0     09/04/2023, Release,  Simone Cimolato
- 1     04/05/2023, update,  Riccardo Cadamuro 
                    the fuction now returns only the complete wind data
                    matrix for the whole set of N simulations

Copyright © 2021, Skyward Experimental Rocketry, AFD department
All rights reserved

SPDX-License-Identifier: GPL-3.0-or-later

%}            
    
    if settings.wind.input
        unc = settings.wind.inputUncertainty;
        uncert = (2*rand(N, 2) - 1) .* unc;

        windMat = uncert; 

    elseif settings.wind.variable
        windG = [normrnd(settings.wind.var.magMeanG, settings.wind.var.magSigmaG, N, 1), ...
                    (settings.wind.var.azMaxG - settings.wind.var.azMinG)*rand(N, 1) + settings.wind.var.azMinG]; 
        hG = settings.wind.var.hG * ones(N, 1); 
        
        windH = [(settings.wind.var.magMaxH - settings.wind.var.magMinH)*rand(N, 1) + settings.wind.var.magMinH, ...
                    wrapTo2Pi((2*rand(N, 1) - 1).*settings.wind.var.deltaAzH + windG(:, 2))];
        hH = settings.wind.var.hH * ones(N, 1);
        
        windMat = [windG, hG, windH, hH];
        
    elseif settings.wind.model
        Day = randi([settings.wind.DayMin, settings.wind.DayMax], N, 1);
        Hour = randi([settings.wind.HourMin, settings.wind.HourMax], N, 1);

        windMat = [Day, Hour]; 
    else
    
        uw = zeros(N, 1); vw = uw; ww = uw; Azw = uw;

        for i = 1:N
            [uw(i), vw(i), ww(i), Azw(i)] = windConstGenerator(settings.wind);
        end

        windMat = [uw,vw, ww, Azw]; 
    end

end