function addTransient(motorName, ignTrans, cutoffTrans, mission)
%
% This function allows you to add an ignition and a cut off transient to the
% thrust curve of an hybrid rocket engine
%
%
arguments
    motorName
    ignTrans        double  % [s/10] ignition transient in tenths of a second
    cutoffTrans     double  % [s/10] cutoff transient in tenths of a second
    mission         Mission
end

if ~contains(motorName, 'HRE')
    error('This function only works with hybrid motors')
end

load(fullfile(mission.dataPath, 'motors.mat'), 'motors');

if ~any(strcmp({motors.MotorName}, motorName))
    error('No motor named motorName found in motors.mat')
end
% currentPath = fileparts(mfilename("fullpath"));
% commonPath = fullfile(currentPath, '..', '..', '..', 'common');
% addpath(commonPath);

% ignTrans = 0.3;     % ignition transient [s]
% coTrans = 0.3;      % cut-off transient  [s]
% mission = Mission('2024_Lyra_Portugal_October');
% name = 'HRE_dummy';
newMotorName = strcat(motorName, '_T0', num2str(ignTrans), '0', num2str(cutoffTrans));
ignTrans = ignTrans/10;
cutoffTrans = cutoffTrans/10;

%% load data
% addpath(mission.dataPath);

if any(strcmp({motors.MotorName}, newMotorName))
    answer = questdlg('This action will overwrite your data. Do you want to continue?','Warning','Yes','No','No');
    switch answer
        case 'Yes'
            motors(strcmp({motors.MotorName}, newMotorName)) = [];
        case 'No'
            error('change motorName')
    end
end

iMotor = strcmp({motors.MotorName}, motorName);
motor = motors(iMotor);

%% Calculations
nT = length(motor.T);

%searching for the firts thrust peak
iPeak =  find(motor.T(2 : end) < motor.T(1 : end - 1), 1, 'first');

dt = (motor.t(end) - motor.t(1))/nT;  % average time step in motor.t array

nTimeIGN = floor(ignTrans/dt);      % number of time elements in ignition transient
expThrustTransIGN = linspace(0, motor.T(iPeak), nTimeIGN);          % ignition transient profile
tIGN = linspace(0, ignTrans, nTimeIGN);                             % ingnition transient time vec

nTimeCO = floor(cutoffTrans /dt);        % number of time elements in cutoff transient
expThrustTransCO = linspace(motor.T(end), 0, nTimeCO);              % cutoff transient profile
tCO = linspace(motor.t(end), motor.t(end) + cutoffTrans, nTimeCO );     % cutoff transient time vec

timeNew = [ tIGN , motor.t(iPeak+1:end-1) + ignTrans, ignTrans + tCO ];     % new thrust curve time vector (with 2 transient
thrustNew = [expThrustTransIGN, motor.T(iPeak+1:end-1), expThrustTransCO];  % new thrust curve
thrustTime = [motor.t, tCO];                                                % actual thrust time vector

timeNew = timeNew * (thrustTime(end)) /timeNew(end);                        % scaling thrust curve time vector in order to have the same duration of thrustTime (actual burnig time)
timeData = motor.t * timeNew(end)/motor.t(end);                             % scaling the original time vector in order to have the correct duration
timeData(end) = timeNew(end);                                               % in order to prevent numerical errors

% Total impulse
Itot = trapz(motor.t, motor.T);
Imod = trapz(timeNew, thrustNew);

% Scaling the thrust curve in order to have the correct total impulse
thrustNew = thrustNew*(Itot/Imod);

% Reshaping the thrust curve in order to have the correct number of elements
thrustNew = interp1(timeNew, thrustNew, timeData);

%% saving data
motor.T = thrustNew;
motor.t = timeData;

nMotor = length(iMotor);
motors(nMotor + 1) = motor;
motors(nMotor + 1).MotorName = newMotorName;
motors(nMotor + 1).LastChange = datetime('now');
motors(nMotor + 1).transientFlag = true;

tableMotors = struct2table(motors);
tableMotors = sortrows(tableMotors,'MotorName');
motors = table2struct(tableMotors);

save(fullfile(mission.dataPath, 'motors.mat'), 'motors');
fprintf(strcat('Transient on motor "',newMotorName,'" successfully added\n'));
end