%{
This script allows you to add an ignition and a cut off transient to the
thrust curve of an hybrid rocket engine
%}

clearvars -except HREmotors
clc
close all

ignTrans = 0.3;     % ignition transient [s]
coTrans = 0.3;      % cut-off transient  [s]

name = 'HRE_ARM_3_new';
newname = strcat(name,'-T03T03mod');

cutPeakFlag = true;

%% load data
filePath = fileparts(mfilename('fullpath'));
currentPath = pwd;
if not(strcmp(filePath, currentPath))
    cd(filePath);
    currentPath = filePath;
end

addpath(genpath(currentPath));

if isunix
    dataPath = '../../data';
else
    dataPath = '..\..\data';
end
addpath(genpath(dataPath))

load('HREmotors.mat')

if any(strcmp({HREmotors.MotorName}, newname))
    answer = questdlg('This action will overwrite your data. Do you want to continue?','Warning','Yes','No','No');
    switch answer
        case 'Yes'
            HREmotors(strcmp({HREmotors.MotorName}, newname)) = [];
        case 'No'
            error('change name')
    end
end

iMotor = strcmp({HREmotors.MotorName}, name);
motor = HREmotors(iMotor);
time = motor.t;
thrust = motor.T;

%% Calcs
timeSteps = length(time);
timeStepSize = (time(end) - time(1))/timeSteps;  % average time step in motor.t array

if cutPeakFlag == true
    % Searching for a value of trhust after the initial peak
    maxSearchTime = 0.1;
    searchLength = 10;
    diffT = diff(thrust);
    meanDiff = mean(diffT(time(2:end) > maxSearchTime));

    currentMean = -inf;
    ind = 0;
    while (-currentMean > -meanDiff) && (time(ind + 1) <= maxSearchTime)
        currentMean = mean(diffT((searchLength * ind) + 1 : searchLength * (ind + 1)));
        ind = ind + 1;
    end

    if (time(ind + 1) > maxSearchTime)
        error('Increase maxSearchTime in order to find a stable point for inital thrust')
    end

    maxInd = ind * searchLength;
else
    %searching for the firts thrust peak
    maxInd = find(thrust(2 : end) < thrust(1 : end - 1), 1, 'first');
end

nTimeIGN = floor(ignTrans / timeStepSize);                          % number of time elements in ignition transient
expThrustTransIGN = linspace(0, thrust(maxInd), nTimeIGN);          % ignition transient profile
tIGN = linspace(0, ignTrans, nTimeIGN);                             % ingnition transient time vec

nTimeCO = floor(coTrans / timeStepSize);                            % number of time elements in cutoff transient
timeMod = [ tIGN , time(maxInd + 1 : end) + ignTrans];      % new thrust curve time vector (with 2 transient
thrustMod = [expThrustTransIGN, thrust(maxInd + 1 : end)];  % new thrust curve

% Searching the time at which the cut off transient should start in order
% to preserve the total impulse with a specified tolerance

Itot = trapz(time, thrust);    % total impulse of initial curve
maxDelta = 4;                  % tolerance on Itot
delta = inf;
lNewTime = length(timeMod);
j = 1;
while delta >= maxDelta && j < lNewTime
    endTimeInd = lNewTime - j;
    expThrustTransCO = linspace(thrustMod(endTimeInd + 1), 0, nTimeCO);
    thrustEvaluated = [thrustMod(1 : endTimeInd), expThrustTransCO];
    tCO = linspace(timeMod(endTimeInd + 1), timeMod(endTimeInd + 1) + coTrans, nTimeCO );
    timeEvaluated = [timeMod(1 : endTimeInd), tCO];
    Imod = trapz(timeEvaluated, thrustEvaluated);

    delta = abs(Imod - Itot);
    j = j + 1;
end

if j == lNewTime
    error('No time step found that satisfies the specified tolerance')
end

% scaling the time vector in order to have the correct total impulse
timeData = time * timeEvaluated(end)/time(end);     % scaling the original time vector in order to have the correct duration
timeData(end) = timeEvaluated(end);                 % in order to prevent numerical errors

% Reshaping the thrust curve in order to have the correct number of elements
thrustData = interp1(timeEvaluated, thrustEvaluated, timeData);

%% saving data
motor.T = thrustData;
motor.t = timeData;

nMotor = length(iMotor);

HREmotors(nMotor+1) = motor;

HREmotors(nMotor+1).MotorName = newname;
HREmotors(nMotor+1).LastChange = datetime('now');
HREmotors(nMotor+1).transientFlag = true;

tableMotors = struct2table(HREmotors);
tableMotors = sortrows(tableMotors,'MotorName');

HREmotors = table2struct(tableMotors);

if isunix
    save('../../data/HREmotors.mat','HREmotors')
else
    save('..\..\data\HREmotors.mat','HREmotors')
end

fprintf(strcat('Transient on motor "',newname,'" successfully added\n'));
