function [input] = createDissileInput(rocket, vars)
%{

createDissileInput - function to create the input file to run dissileMatcom

INPUTS:
        - rocket, Rocket object, containing the rocket's details
        - vars, struct, containing the simulation's details
                        - Mach  [1, NMACH] : contains the mach numbers to be computed (NMACH <= 20)
                        - Alpha [1, NALPHA]: contains the alpha values to be computed [deg]
                                             the array shall be symmetric wrt 0 and contain -1, 0, 1
                        - Beta  [1, NBETA] : contains the beta values to be computed [deg]
                        - Alt   [1, NALT]  : contains the altitude values to be computed [m]
                        - xcg   [1, nxcg]  : contains the xcg to be computed, from the nosecone base [m]
                        - hprot [1, nhprot]: contains the protuberance height to be computed [m]
                                             insert empty array [] if no protuberance shall be computed 

OUTPUTS:
        - input, struct, input struct used by dissileMatcom to calculate 
                 the aerodynamic coefficients

CALLED FUNCTIONS: -

CALLED SCRIPTS: -

REVISION:
- #0    23/08/2022, Release, Simone Cimolato
- #1    13/02/2023, Update,  Riccardo Cadamuro, Maria Teresa Cazzola
                             function suitable for generic usage

%}

arguments
    rocket Rocket
    vars
end

%% CHECK ERROR

if length(vars.alpha) > 20 || length(vars.mach) > 20
    error("dissileMatcom with alpha or mach > 20 cases is not yet compatible");
end

if not(any(vars.alpha == 0))
    error('vars.Alpha does not contains 0');
end

if not(any(vars.alpha == 1))
    error('vars.Alpha does not contains 1');
end

if not(any(vars.alpha == -1))
    error('vars.Alpha does not contains -1');
end

if not(isequal(vars.alpha, -fliplr(vars.alpha)))
    error('vars.Alpha is not symmetric');
end

noses = {'CONICAL', 'OGIVE', 'POWER', 'HAACK', 'KARMAN', 'MHAACK'};

%% FLTCON
% Flight conditions quantities
input.fltcon.about = 'Flight conditions quantities';
input.fltcon.MACH = vars.mach;
input.fltcon.ALPHA = vars.alpha;
input.fltcon.BETA = vars.beta;
input.fltcon.ALT = vars.alt;

%% REFQ
% Reference quantities
input.refq.about = 'Reference quantities';
input.refq.XCG = vars.xcg;                                                                       % [m] Longitudinal position of C.G. (distance between C.G. and the base of the nose cone, LNOSE will be added later in dissileMatcom/GEOM)
input.refq.SREF = rocket.crossSection;                                                           % [m^2] Reference area
input.refq.LREF = rocket.diameter;                                                               % [m] Longitudinal reference length
input.refq.LATREF = rocket.diameter;                                                             % [m] Lateral reference length

%% AXIBOD
% Axisymmetric bodies quantities
input.axibod.about = 'Axisymmetric body quantities';

%%% nose
input.axibod.TNOSE = find(strcmp(noses, rocket.payload.noseType))-1;                             % [0=CONICAL, 1=OGIVE, 2=POWER, 3=HAACK, 4=KARMAN, 5=MHAACK] ogive type
if strcmp(rocket.payload.noseType, 'MHAACK')
    input.axibod.MHAACKCP = [rocket.payload.noseCMod rocket.payload.nosePMod];                   % [-] CMOD PMOD mhaack parameters, set CMOD = 1/3 and PMOD = 1 to get a haack
elseif strcmp(rocket.OgType, 'POWER')
    input.axibod.POWER = rocket.payload.nosePower;                                               % [-] Exponent of the power series, set a value between 0 and 1
end
input.axibod.LNOSE = rocket.payload.noseLength;                                                  % [m] Nose length
input.axibod.DNOSE = rocket.diameter;                                                            % [m] Nose diameter at base
input.axibod.TRUNC = false;

%%% pitot
if ~isempty(rocket.pitot.initialConeLength)
    input.axibod.PITOTCLI = rocket.pitot.initialConeLength;                                      % [m] Pitot initial conic section length
    input.axibod.PITOTCLB = rocket.pitot.finalConeLength;                                        % [m] Pitot final conic section length
    input.axibod.PITOTCDI = rocket.pitot.initialConeDiameter;                                    % [m] Pitot initial conic section diameter
    input.axibod.PITOTCDB = rocket.pitot.finalConeDiameter;                                      % [m] Pitot final conic section diameter
    input.axibod.PITOTL = rocket.pitot.length;                                                   % [m] Pitot tube length
    input.axibod.PITOTD = rocket.PITOTD;                                                         % [m] Pitot tube diameter
else
    input.axibod.PITOTCLI = 0;                                                                  % [m] Pitot initial conic section length
    input.axibod.PITOTCLB = 0;                                                                  % [m] Pitot final conic section length
    input.axibod.PITOTCDI = 0;                                                                  % [m] Pitot initial conic section diameter
    input.axibod.PITOTCDB = 0;                                                                  % [m] Pitot final conic section diameter
    input.axibod.PITOTL = 0;                                                                    % [m] Pitot tube length
    input.axibod.PITOTD = 0;                                                                    % [m] Pitot tube diameter
end

%%% centerbody
input.axibod.LCENTR = rocket.lengthCenter;                                                       % [m] Centerbody length
input.axibod.DCENTR = rocket.diameter;                                                           % [m] Centerbody diameter at base

%%% boattail
if (~isempty(rocket.rear.boatType))
    boats = {'CONICAL', 'OGIVE'};
    input.axibod.LAFT = rocket.rear.boatLength;                                                  % [m] Boattail length
    input.axibod.DAFT = rocket.rear.boatFinalDiameter;                                           % [m] Boattail base diameter
    input.axibod.TAFT = find(strcmp(boats, rocket.rear.boatType))-1;                                                    % [-] Boattail type
end
input.axibod.DEXIT = 0;                                                                          % [m] Nozzle diameter for base drag calculation
input.axibod.BASE = false;                                                                       % [-]

%%% protub
if ~isempty(vars.hprot)
    input.axibod.PROTUB = true;                                                                  % Flag if protub exist
    input.axibod.NPROT = 1;                                                                      % [-] N° set protub
    input.axibod.PTYPE = 'BLOCK';                                                                % Protub type (only block exist)
    input.axibod.XPROT = rocket.airbrakes.xDistance;                                             % [m] Longitudinal distance from missile nose cone base to the geometric centroid of the protuberance set (if there is a pitot tube LNOSE will be added later in dissileMatcom/GEOM)
    input.axibod.NLOC = rocket.airbrakes.n;                                                      % [-] N° protub in each set
    input.axibod.LPROT = rocket.airbrakes.thickness;                                             % [m] Thickness of protuberance
    input.axibod.WPROT = rocket.airbrakes.width;                                                 % [m] Width of protuberance
    input.axibod.HPROT = vars.hprot;                                                             % [m] Height of protuberance
    input.axibod.OPROT = 0;                                                                      % [m] Vertical offset of protuberance
else
    input.axibod.PROTUB = false;
end

%% FINSET1
%%% Fin set 1 properties
input.finset1.about = 'Finset 1 properties';
input.finset1.XLE = [rocket.lengthCenter - rocket.rear.finsAxialDistance - ...
    rocket.rear.finsRootChord, rocket.lengthCenter - rocket.rear.finsAxialDistance - ...
    rocket.rear.finsRootChord + rocket.rear.finsDeltaXFreeChord];                                % [m] Distance from missile nose cone base to chord leading edge at each span location (LNOSE will be added later in dissileMatcom/GEOM)
input.finset1.NPANEL = rocket.rear.nPanel;                                                       % [-] Number of panels in fin set
input.finset1.PHIF = ...
    (0:1:(rocket.rear.nPanel - 1)) * 360/rocket.rear.nPanel;                                         % [deg] Roll angle of each fin measured clockwise from top vertical center looking forward
input.finset1.LER = rocket.rear.finsLeadingEdgeRadius;                                           % [m] Leading edge radius at each span station
input.finset1.SSPAN = [0, rocket.rear.finsHeight];                                               % [m] Semi-span locations
input.finset1.CHORD = [rocket.rear.finsRootChord, rocket.rear.finsFreeChord];                    % [m] Panel chord at each semi-span location
input.finset1.SECTYP = 0;                                                                        % [0=HEX, 1=NACA, 2=ARC, 3=USER] Airfoil section type
input.finset1.ZUPPER = [rocket.rear.finsSemiThickness/rocket.rear.finsRootChord, ...
    rocket.rear.finsSemiThickness/rocket.rear.finsFreeChord];                                    % [-] Thickness-to-chord ratio of upper surface
input.finset1.LMAXU = [rocket.rear.finsMaxThicknessPosition/rocket.rear.finsRootChord, ...
    rocket.rear.finsMaxThicknessPosition/rocket.rear.finsFreeChord];                                 % [-] Fraction of chord from leading edge to maximum thickness of upper surface
input.finset1.LFLATU = [(rocket.rear.finsRootChord - 2*rocket.rear.finsMaxThicknessPosition)/rocket.rear.finsRootChord,...
    (rocket.rear.finsFreeChord - 2*rocket.rear.finsMaxThicknessPosition)/rocket.rear.finsFreeChord]; % [-] Fraction of chord of constant thickness section of upper surface

%% DO NOT MODIFY
input.fltcon.NALPHA = length(input.fltcon.ALPHA);                                                % [-] Number of alphas
input.fltcon.NMACH = length(input.fltcon.MACH);                                                  % [-] Number of Machs

%% OPTIONS
input.printTxt = false;                                                                          % if you want the coefficients printed in the command window
input.draw = false;                                                                              % if you want the shape of your rocket plotted
input.HREflag = rocket.motor.isHRE;
input.axibod.precision = 11;                                                                     % [-] 1-1001, how many points to approximate nose cone shape, default is 11 (NOTE: POWER SHAPE REQUIRES <= 401)
end
