function [coeffsValues, angle0] = interpCoeffs(coefficients, t, cutoffTime, alpha, mach, beta, alt, c)
% interpCoeffs - interpolation of aerodynamic coefficients.
%
% INPUTS:
%         - t, double [1,1], integration time, [s];
%         - alpha, double[1,1], angle of attack, [];
%         - mach, double[1,1], mach number, [];
%         - beta, double[1,1], sideslip angle, [];
%         - alt, double[1,1], altitude, [m];
%         - c, double[1,1], aerobrakes control variable, [];
%
% OUTPUTS:
%         - coeffsValues, array [16,1],  aerodynamic coefficients;
%         - angle0, array [2,1], array of the reference aerodynamic angles.

h = c*coefficients.state.hprot(end);

[nearAlpha, indexAlpha] = nearestValSorted(coefficients.state.alphas*pi/180, alpha);
[~, indexMach] = nearestValSorted(coefficients.state.machs, mach);
[nearBeta, indexBeta] = nearestValSorted(coefficients.state.betas*pi/180, beta);
[~, indexAlt] = nearestValSorted(coefficients.state.altitudes, alt);
[~, indexControl] = nearestValSorted(coefficients.state.hprot, h);

angle0 = [nearAlpha; nearBeta];

if t >= cutoffTime
    % Interpolate on airbrake control
    % Uses end xCg, even if tCutOff < tBurnTime
    num = length(coefficients.state.hprot);
    index1 = indexControl;
    index2 = indexControl + 1;
    if index2 > num
        if num == 1
            coeffsValues = coefficients(:, indexAlpha, indexMach, indexBeta, indexAlt, 1, end);
            return;
        else
            index2 = index2 - 1;
            index1 = index1 - 1;
        end
    end

    VF = coefficients.total(:, indexAlpha, indexMach, indexBeta, indexAlt, index1, end);
    VE = coefficients.total(:, indexAlpha, indexMach, indexBeta, indexAlt, index2, end);

    deltaH = coefficients.state.hprot(index2) - coefficients.state.hprot(index1);
    coeffsValues =  ( (h - coefficients.state.hprot(index1))/deltaH)*(VE - VF) + VF;
else
    % Interpolate on xCg, close airbrakes
    num = 1:1:length(coefficients.state.xcgTime)-1;
    indexXcg = t >= coefficients.state.xcgTime(1:end-1) & t < coefficients.state.xcgTime(2:end);
    index1 = num(indexXcg);
    index2 = index1 + 1;

    VF = coefficients.total(:, indexAlpha, indexMach, indexBeta, indexAlt, indexControl, index1);
    VE = coefficients.total(:, indexAlpha, indexMach, indexBeta, indexAlt, indexControl, index2);

    deltaT = coefficients.state.xcgTime(index2) - coefficients.state.xcgTime(index1);
    coeffsValues =  ( (t - coefficients.state.xcgTime(index1))/deltaT)*(VE - VF) + VF;
end
end