function [coeffsValues, angle0] = interpCoeffs_old(t, alpha, M, beta, alt, c, settings)
%{
interpCoeffs_old - interpolation of aerodynamic coefficients.

INPUTS:
        - t, double [1,1], integration time, [s];
        - alpha, double[1,1], angle of attack, [];
        - M, double[1,1], mach number, [];
        - beta, double[1,1], sideslip angle, [];
        - alt, double[1,1], altitude, [m];
        - c, double[1,1], aerobrakes control variable, [];
        - alphaTot, double[1,1], total angle of attack, [];
        - settings, struct, rocket and simulation data.

OUTPUTS:
        - coeffsValues, array [16,1],  aerodynamic coefficients;
        - angle0, array [2,1], array of the reference aerodynamic angles.

CALLED FUNCTIONS: -

VERSIONS:
        -0   02/02/2018,   release,    Adriano Filippo Inno
        -1   12/01/2020,   update,     Giulio Pacifici
                        all coefficients inteprolation
        -2   06/11/2021,   update,     Adriano Filippo Inno
                        faster vectorization
Copyright © 2021, Skyward Experimental Rocketry, AFD department
All rights reserved

SPDX-License-Identifier: GPL-3.0-or-later

%}

[nearAlpha, indexAlpha] = nearestValSorted(settings.Alphas*pi/180, alpha);
[~, indexMach] = nearestValSorted(settings.Machs, M);
[nearBeta, indexBeta] = nearestValSorted(settings.Betas*pi/180, beta);
[~, indexAlt] = nearestValSorted(settings.Altitudes, alt);

VE = settings.CoeffsE(:, indexAlpha, indexMach, indexBeta, indexAlt, c);

if t <= settings.tb
    VF = settings.CoeffsF(:, indexAlpha, indexMach, indexBeta, indexAlt);
    coeffsValues =  t/settings.tb*(VE - VF) + VF;
else
    coeffsValues = VE;
end

angle0 = [nearAlpha; nearBeta];
