function [time, launchpad, apogee, c] = singleGArun(lb, ub, input, settings)
%{
singelGArun - estimate the time that a single run of GA scripts takes
                  on your computer. Furthermore, the script is used to
                  compute launchpad dynamics of a i-th rocket aerodynamic
                  configuration; these parameters are stored in launchpad
                  struct.

INPUTS:
        - lb, double [1,7], lower boundaries, [-];
        - ub, double [1,7], upper boundaries, [-];
        - input, struct, datcom variables;
        - settings, struct, rocket and simulation data.

OUTPUTS:
        - time, double [1,1], computational time, [s];
        - launchpad, struct, launchpad dynamics parameters.
        - c, double [1, 1], non linear constrain of the optimization.

CALLED SCRIPTS: -

CALLED FUNCTIONS: OptimizationGA, XCPcheck

VERSIONS:
- 0     31/10/2021, Release,    Davide Rosato
- 1     10/12/2022, Update,     Riccardo Cadamuro, Maria Teresa Cazzola
                                -Added wind cases at 60,120,180 deg azimuth
                                -Initial rocket roll angle = 180deg instead
                                of 0deg according to Missile Datcom
                                reference frame

Copyright © 2021, Skyward Experimental Rocketry, AFD department
All rights reserved

SPDX-License-Identifier: GPL-3.0-or-later

%}

tic

%% SET VARIABLES
% the variables are set equal to the mean values of their upper and lower
% bundaries
N = length(lb);
x = zeros(1, N);

for i = 1:N
    x(i) = round((ub(i) + lb(i)) / 2);
end

%% XCPcheck

%%%  RETRIVING DATCOM VARS FROM THE OPTIMIZATION VARIABLES
% x is looping
[input] = createDatcomStruct(x, settings, input);

%%% AERODYNAMICS COEFFICIENT - LAUNCHPAD
% this section is computed just once, then the resulting axial coefficient
% is kept costant. This consists in a small approximation on the final
% result but the computational time is reduced by 5/10%

%%% states to compute the exit pad velocity
input.fltcon.MACH = 0.1;
input.fltcon.ALPHA = [-1, -0.5, 0, 0.5, 1];
input.fltcon.BETA = 0;
input.fltcon.ALT = settings.z0;
input.fltcon.NALPHA = length(input.fltcon.ALPHA);   % [-] Number of alphas
input.fltcon.NMACH = length(input.fltcon.MACH);

xcg = settings.xcg;
input.refq.XCG = xcg(1);

[coeff] = dissileMatcom(input);
Coeffs0 = coeff(:,:,:,:,:,1);
CA0 = Coeffs0(1, 3);

%%% LAUNCHPAD DYNAMICS
%%% Initial State
X0pad = [0; 0; 0; 0];
%%% Attitude
Q0 = angleToQuat(settings.PHI, settings.OMEGA, 180*pi/180)';

[Tpad, Ypad] = ode45(@launchPadFreeDyn, [0, 10], X0pad, settings.ode.optionspad,...
    settings, Q0, CA0);
launchpad.Tpad = Tpad;

%%% COMPUTING THE LAUNCHPAD STABILITY DATA
% both lateral and longitudinal XCP involved
% the launchpad dynamics is used to write the aerodyanmics states
T = (288.15 - 0.0065*settings.z0);       % temperature
a = sqrt(T*1.4*287.0531);                % sound speed @launchpad
input.fltcon.MACH = round(Ypad(end, 4)/a, 2);  % Mach @launchpad
input.fltcon.ALT = settings.z0;                % Altitude @launchpad
launchpad.datcom.Mach = input.fltcon.MACH;

% Wind directions to be looped symmetrical wrt to the launch
% note: 180 deg is discarded because 0 deg is ever a worst case scenario)
Az = [0 45 60 90 120 135 180]*pi/180;

%%% pre-allocation
nAz = length(Az); 
alphaExit = zeros(1, nAz);
betaExit = zeros(nAz, 1);

for i = 1:nAz
    settings.wind.Az = Az(i);            % set the direction
    [uw, vw, ww] = windConstGenerator(settings.wind);
    inertialWind = [uw, vw, ww];
    bodyWind = quatrotate(Q0', inertialWind);
    bodyVelocity = [Ypad(end, 4), 0, 0];
    Vr = bodyVelocity - bodyWind;
    ur = Vr(1); vr = Vr(2); wr = Vr(3);
    alphaExit(i) = round(atand(wr/ur), 1);
    betaExit(i) = round(atand(vr/ur), 1);
end
launchpad.betaExit = unique(betaExit);

%%% alpha vector imposed to be symmetric and populated
alphaVectorPositive = [abs(alphaExit) 1];
alphaVectorPositive = sort(alphaVectorPositive);
input.fltcon.ALPHA = unique(sort([-alphaVectorPositive, 0, alphaVectorPositive]));
input.fltcon.BETA = sort(launchpad.betaExit);
input.fltcon.NALPHA = length(input.fltcon.ALPHA);   % [-] Number of alphas
input.fltcon.NMACH = length(input.fltcon.MACH);
launchpad.datcom.Alpha = input.fltcon.ALPHA;
launchpad.datcom.Beta = input.fltcon.BETA;

%%% aerodynmics coefficient - full
input.refq.XCG = xcg([1, length(xcg)]);

for i = 1:nAz
    indexAlpha = find(alphaExit(i) == input.fltcon.ALPHA);
    indexBeta = find(betaExit(i) == input.fltcon.BETA);
    launchpad.indexAlpha(i) = indexAlpha-1;
    launchpad.indexBeta(i) = indexBeta;
end
settings.launchpad = launchpad;
[c, ~] = XCPcheck(x, input, settings);

%% OptimizationGA
[apogee] = OptimizationGA(x, input, settings);
time = toc;

end