function [dY, parout] = ballistic(t, Y, rocket, environment, wind, control, t0, wrapper)
arguments
    t
    Y
    rocket          Rocket
    environment     Environment
    wind            % WindCustom {mustBeA(wind, {'WindCustom', 'WindMatlab'})}
    control         struct = []
    t0              double = 0
    wrapper                = [] %DataWrapper = DataWrapper.empty
end
% ballistic - ode function of the 6DOF Rigid Rocket Model
% 
% INPUTS:
% - t,       double [1, 1] integration time [s];
% - Y,       double [14, 1] state vector [ x y z | u v w | p q r | q0 q1 q2 q3 | angle]:
%                                   * (x y z), NED{north, east, down} horizontal frame;
%                                   * (u v w), body frame velocities;
%                                   * (p q r), body frame angular rates;
%                                   * (q0 q1 q2 q3), attitude unit quaternion;
%                                   * (angle), airbrake servo angle
%
% - (rocket, environment, wind), data Classes
% - control, struct (angleRef, timeChangeRef, partialTime) GNC parameters:
%                                   * angleRef, airbrake servo angle reference
%                                   * timeChangeRef, time to change airbrake configuration
%                                   * partialTime, 
% - wrapper, handle to export data
% 
% OUTPUTS:
% - dY,      double [14, 1] state derivatives
% - parout,  struct, additional flight qunatities
%
% Copyright © 2021, Skyward Experimental Rocketry, AFD, GNC departments
% All rights reserved
% 
% SPDX-License-Identifier: GPL-3.0-or-later

%% RECALLING STATES
% x = Y(1);
% y = Y(2);
altitude = -Y(3);                                                           % z is directed as the gravity vector (NED reference)
u = Y(4);
v = Y(5);
w = Y(6);
p = Y(7);
q = Y(8);
r = Y(9);
qw = Y(10);
qx = Y(11);
qy = Y(12);
qz = Y(13);
angle = Y(14);

dY = zeros(14, 1);
% if altitude < 0, altitude = 0; end

%% GETTING DATA
S = rocket.crossSection;                                                    % [m^2]   cross surface
C = rocket.diameter;                                                        % [m]     caliber
g = environment.g0/(1 + (altitude*1e-3/6371))^2;                            % [N/kg]  module of gravitational field
tb = rocket.motor.cutoffTime;                                               % [s]     Burning Time
theta = t-t0;                                                               % [s]     Time shift (needed for control)
local = environment.local;                                                  % vector containing inputs for atmosphereData
omega = environment.omega;

isControlActive = false;
isAngleSaturated = false;

%% QUATERION ATTITUDE
Q = [qw qx qy qz];
Q = Q/norm(Q);

%% ADDING WIND (supposed to be added in NED axes);
switch class(wind)
    case 'WindMatlab'
        [uw, vw, ww] = wind.getVels(altitude, theta);
    case 'WindCustom'
        [uw, vw, ww] = wind.getVels(altitude);
end

dcm = quatToDcm(Q);
windVels = dcm*[uw; vw; ww];

% Relative velocities (plus wind);
ur = u - windVels(1);
vr = v - windVels(2);
wr = w - windVels(3);


vels = dcm'*[u; v; w];                                                      % Body to Inertial velocities
velsNorm = norm([ur vr wr]);

%% ATMOSPHERE DATA
absoluteAltitude = altitude + environment.z0;
[~, a, P, rho] = atmosphereData(absoluteAltitude, g, local);

mach = velsNorm/a;

%% TIME-DEPENDENTS VARIABLES
if theta < tb
    m = interpLinear(rocket.motor.time, rocket.mass, theta);
    T = interpLinear(rocket.motor.time, rocket.motor.thrust, theta);
    Pe = interpLinear(rocket.motor.time, rocket.motor.pe, theta);
    T = T + rocket.motor.ae*(Pe - P);

    I = interpLinear(rocket.motor.time, rocket.inertia, theta);
    Idot = interpLinear(rocket.motor.time, rocket.inertiaDot, theta);
else                                                                        % for theta >= tb the flight condition is the empty one (no interpolation needed)
    m = rocket.cutoffMass;
    T = 0;

    I = rocket.cutoffInertia;
    Idot = zeros(3, 1);
end

Ixx = I(1); Iyy = I(2); Izz = I(3);
Ixxdot = Idot(1); Iyydot = Idot(2); Izzdot = Idot(3);

%% CONTROL VARIABLES

if isempty(control)
    % MSA simplified control
    extension = rocket.airbrakes.getExtension(theta, mach);
else
    isControlActive = true;
    if angle > rocket.airbrakes.servoMaxAngle
        angle = settings.servo.maxAngle;
        isAngleSaturated = true;
    elseif angle < rocket.airbrakes.servoMinAngle
        angle = settings.servo.minAngle;
        isAngleSaturated = true;
    end
    extension = rocket.airbrakes.angle2Extension(angle);
end

%% AERODYNAMICS ANGLES
if not(abs(ur) < 1e-9 || velsNorm < 1e-9)
    alpha = atan(wr/ur);
    beta = atan(vr/ur);                                                     % beta = asin(vr/V_norm) is the classical notation, Datcom uses this one though.
    % alpha_tot = atan(sqrt(wr^2 + vr^2)/ur);   % datcom 97' definition
else
    alpha = 0;
    beta = 0;
end

alphaOut = alpha;
betaOut = beta;

%% INTERPOLATE AERODYNAMIC COEFFICIENTS:

if abs(alpha)>25*pi/180 || abs(beta)>25*pi/180
    coeffsValues = interpN( rocket.coefficientsHighAOA.total,...
        {rocket.coefficientsHighAOA.state.alphas, rocket.coefficientsHighAOA.state.machs, ...
        rocket.coefficientsHighAOA.state.betas, rocket.coefficientsHighAOA.state.altitudes}, ...
        [alpha, mach, beta, absoluteAltitude]);
    angle0 = [alpha beta];
else
    %[coeffsValues, angle0] = rocket.interpCoeffs(theta, alpha, mach, beta, absoluteAltitude, opening);
    [coeffsValues, angle0] = interpCoeffs(rocket.coefficients, theta, rocket.motor.cutoffTime, ...
        alpha, mach, beta, absoluteAltitude, extension);
end

% Retrieve Coefficients
CA = coeffsValues(1); CYB = coeffsValues(2); CY0 = coeffsValues(3);
CNA = coeffsValues(4); CN0 = coeffsValues(5); Cl = coeffsValues(6);
Clp = coeffsValues(7); Cma = coeffsValues(8); Cm0 = coeffsValues(9);
Cmad = coeffsValues(10); Cmq = coeffsValues(11); Cnb = coeffsValues(12);
Cn0 = coeffsValues(13); Cnr = coeffsValues(14); Cnp = coeffsValues(15);

% Clb = coeffsValues(16);
% XCP_value = coeffsValues(16);

% compute CN,CY,Cm,Cn (linearized with respect to alpha and beta):
alpha0 = angle0(1); beta0 = angle0(2);

CN = (CN0 + CNA*(alpha - alpha0));
CY = (CY0 + CYB*(beta - beta0));
Cm = (Cm0 + Cma*(alpha - alpha0));
Cn = (Cn0 + Cnb*(beta - beta0));

% XCPlon = Cm/CN;
if abs(alpha) <= pi/180
    XCPlon = Cma/CNA;
else
    XCPlon = Cm/CN; 
end
 
% XCPlat = Cn/CY;
if abs(beta) <= pi/180
    XCPlat = Cnb/CYB; 
else
    XCPlat = Cn/CY; 
end

% if Cn == 0 && CY == 0
%     XCPlat = -5;
% end

%% RAMP / FREE FLIGHT
if altitude < environment.effectiveRampAltitude                             % No torque on the launchpad
    Fg = m*g*sin(omega);                                                    % [N] force due to the gravity
    fX = 0.5*rho*velsNorm^2*S*CA;
    F = -Fg +T -fX;
    du = F/m;

    dv = 0;
    dw = 0;
    dp = 0;
    dq = 0;
    dr = 0;
    dAngle = 0;

    alphaOut = NaN;
    betaOut = NaN;
    fY = 0;
    fZ = 0;
    XCPlon = NaN;
    XCPlat = NaN;

    if T < Fg                                                               % No velocity untill T = Fg
        du = 0;
    end
    
    XCPtot = NaN; 
else
%% FORCES
    % first computed in the body-frame reference system
    qdyn = 0.5*rho*velsNorm^2;                                              % [Pa] dynamics pressure
    qdynL_V = 0.5*rho*velsNorm*S*C;

    fX = qdyn*S*CA;                                                         % [N] x-body component of the aerodynamics force
    fY = qdyn*S*CY;                                                         % [N] y-body component of the aerodynamics force
    fZ = qdyn*S*CN;                                                         % [N] z-body component of the aerodynamics force
    Fg = dcm*[0; 0; m*g];                                                   % [N] force due to the gravity in body frame
    F = Fg + [-fX+T, fY, -fZ]';                                             % [N] total forces vector
    
    %-----------------------------------------------------
    %F = Fg + [-X+T*cos(chi), Y+T*sin(chi), -Z]';             % [N] total forces vector
    %-----------------------------------------------------

%% STATE DERIVATIVES
    % velocity
    du = F(1)/m - q*w + r*v;
    dv = F(2)/m - r*u + p*w;
    dw = F(3)/m - p*v + q*u;

    % Rotation
    dp = (Iyy - Izz)/Ixx*q*r + qdynL_V/Ixx*(velsNorm*Cl+Clp*p*C/2) - Ixxdot*p/Ixx;
    dq = (Izz - Ixx)/Iyy*p*r + qdynL_V/Iyy*(velsNorm*Cm + (Cmad+Cmq)*q*C/2)...
        - Iyydot*q/Iyy;
    dr = (Ixx - Iyy)/Izz*p*q + qdynL_V/Izz*(velsNorm*Cn + (Cnr*r+Cnp*p)*C/2)...
        - Izzdot*r/Izz;

    % Compute the aerodynamic roll angle 
    [~, phi] = getAlphaPhi(alpha, beta); 
        
    % Aerodynamic-force coefficient in the alpha-total plane
    CFaTot = sin(phi)*CY + cos(phi)*(-CN);      
    % Aerodynanic-moment coefficient in the alpha-total plane
    CMaTot = cos(phi)*Cm - sin(phi)*Cn;
    if CFaTot ~= 0
        XCPtot = CMaTot/CFaTot; 
    else
        XCPtot = XCPlon; 
    end

    dAngle = 0;
    if isControlActive && ~isAngleSaturated
        angleRef = control.angleRef;
        if ~rocket.airbrakes.identification
            % set velocity of servo (air brakes)
            if length(angleRef) == 2                                    % for the recallOdeFunction
                if theta < control.timeChangeRef
                    angleRef = angleRef(1);
                else
                    angleRef = angleRef(2);
                end
            else
                [~,idxAirbrake] = min(control.partialTime - theta);
                angleRef = angleRef(idxAirbrake);                       % don't delete this unless you change how the recall ode works.
            end
        else
            [~,idxAirbrake] = min(abs(theta-angleRef(:,1)));                % if we are trying to identify we need to have the same input of the flight
            angleRef = angleRef(idxAirbrake,2);
        end

        dAngle = (angleRef-angle)/rocket.airbrakes.servoTau;
        if abs(dAngle) > rocket.airbrakes.servoOmega
            dAngle = sign(angleRef-angle)*rocket.airbrakes.servoOmega;      % concettualmente sta roba è sbagliata perchè dipende dal passo di integrazione, fixare
        end
    end
end
% Quaternions
OM = [ 0 -p -q -r  ;
       p  0  r -q  ;
       q -r  0  p  ;
       r  q -p  0 ];

dQQ = 1/2*OM*Q';

%% FINAL DERIVATIVE STATE ASSEMBLING
dY(1:3) = vels;
dY(4:6) = [du; dv; dw];
dY(7:9) = [dp; dq; dr];
dY(10:13) = dQQ;
dY(14) = dAngle;

%% SAVING THE QUANTITIES FOR THE PLOTS

if nargout == 2 || ~isempty(wrapper)
    parout.state = [];

    parout.interp.mach = mach;
    parout.interp.alpha = alphaOut;
    parout.interp.beta = betaOut;
    parout.interp.alt = altitude;
    parout.interp.mass = m;
    parout.interp.inertias = [Ixx; Iyy; Izz]; 

    parout.wind.NED = [uw; vw; ww];
    parout.wind.body = windVels;
    
    parout.rotations.dcm = dcm;
    
    parout.velocities = vels;
    
    parout.forces.aero = [fX; fY; fZ];
    parout.forces.T = T;
    
    parout.air.rho = rho;
    parout.air.P = P;
    
    parout.accelerations.body = [du; dv; dw];
    parout.accelerations.angular = [dp; dq; dr];
    
    parout.coeff.CA = CA;
    parout.coeff.CYB = CYB;
    parout.coeff.CNA = CNA;
    parout.coeff.Cl = Cl;
    parout.coeff.Clp = Clp;
    %parout.coeff.Clb = Clb;
    parout.coeff.Cma = Cma;
    parout.coeff.Cmad = Cmad;
    parout.coeff.Cmq = Cmq;
    parout.coeff.Cnb = Cnb;
    parout.coeff.Cnr = Cnr;
    parout.coeff.Cnp = Cnp;
    parout.coeff.XCPlon = XCPlon;
    parout.coeff.XCPlat = XCPlat;
    parout.coeff.XCPtot = XCPtot;
    
    parout.uncertanty = [];
    if ~isempty(wrapper)
        wrapper.setCache(parout); 
    end
end