function data_para = stochDescentLoop(settings, data_ascent)
%{
stochDescentLoop - This function runs the descent inside the parfor loop

INPUTS:     
            - settings,     struct: stores data of the rocket and of the simulation.
            - data_ascent,  struct: ascent simulations data to compute the parachute
                                   descent initial conditions

OUTPUTS:
            - data_para,cell [1, k]: cell containg the struct with the flght data of 
                                     each stage of the rochet during the parachute
                                     descent

CALLED FUNCTIONS: 

REVISIONS:
- 0     17/03/2023, Release,  Simone Cimolato
- 1     04/03/2023, update,   Riccardo Cadamuro
                    all data saved in settings except for the initial
                    conditions stored in data_ascent

Copyright © 2021, Skyward Experimental Rocketry, AFD department
All rights reserved

SPDX-License-Identifier: GPL-3.0-or-later
%}

    tf = settings.ode.finalTime;
    Ya = data_ascent.state.Y;
    
    data_para = cell(1, settings.NdescentStages);
    
    for k = 1:settings.NdescentStages
        Y0p = [Ya(end, 1:3) quatrotate(quatconj(Ya(end, 10:13)),Ya(end, 4:6))];
        t0p = data_ascent.state.T(end);

        data_para{k}.para = cell(settings.Npara(k), 1);

        for ii = 1:settings.Npara(k)
            if ii == 1
                YaEndTrue = Ya(end, 4:6) + [settings.para(ii, k).Vexit 0 0];
                Y0p = [Y0p(1:3) quatrotate(quatconj(Ya(end, 10:13)), YaEndTrue)];
            end
            para = ii; settings.stoch.para = para; settings.stage = k;
            [Tp, Yp] = ode113(@descentParachute, [t0p, tf], Y0p, settings.ode.optionspara, settings);

            [data] = recallOdeFcn(@descentParachute, Tp, Yp, settings);
            data_para{k}.para{ii} = data;
            data_para{k}.para{ii}.state.Y = Yp;
            data_para{k}.para{ii}.state.T = Tp;

            % updating ODE starting conditions
            Y0p = Yp(end, :);
            t0p = Tp(end);
        end
    end

    if settings.parafoilSim 
        % Open loop parafoil dynamics 
        ind1 = settings.payload.paraPrev(1);
        ind2 = settings.payload.paraPrev(2); 
        t0Parafoil  = data_para{ind1}.para{ind2}.state.T(end); 
        posVel0Parafoil = data_para{settings.payload.paraPrev(1)}.para{settings.payload.paraPrev(2)}.state.Y(end, :); 
        
        windNED = data_para{ind1}.para{ind2}.wind.NED_wind(:, end); 
    
        az = atan2(windNED(2), windNED(1)); 
        el = asin(-windNED(3)/norm(windNED(1:2))); 
        quat0Parafoil = angleToQuat(az, el, 0);
        
        initPos = posVel0Parafoil(1:3); 
        initVelBody = quatrotate(quat0Parafoil, posVel0Parafoil(4:6)); 
    
        Y0Parafoil = [initPos, initVelBody, 0, 0, 0, quat0Parafoil]'; 
    
        [TParafoil, YParafoil] = ode113(@descentParafoil, [t0Parafoil, tf], Y0Parafoil, settings.ode.optionsdesc, settings); 
        
        [data] = recallOdeFcn(@descentParafoil, TParafoil, YParafoil, settings); 
    
        data_para{ind1}.para{ind2+1} = data;    
        data_para{ind1}.para{ind2+1}.state.T = TParafoil; 
        data_para{ind1}.para{ind2+1}.state.Y = YParafoil; 
    end
    
end