    function data_stability = stochStabilityLoop(settings, Y0)
%{
stochStabilityLoop - This function runs a simulation to compute the rocket
                     stability inside the parfor loop

INPUTS:     
            - settings,     struct: stores data of the rocket and of the simulation.
            - Y0,   double [13, 1]: initial state vector of the integration 
                                    [x, y, z, u, v, w, p, q, r, q0, q1, q2, q3]

OUTPUTS:
            - data_stability,cell [1, k]: cell containg the struct with the rocket 
                                          stability data at the launchpad exit

CALLED FUNCTIONS: 

REVISIONS:
- 0     14/06/2023, Release,    Riccardo Cadamuro

Copyright © 2023, Skyward Experimental Rocketry, AFD department
All rights reserved

SPDX-License-Identifier: GPL-3.0-or-later
%}
    
    %% comupute launchpad run
    [Tpad, Ypad] = ode113(@ascent, [0, 10], Y0, settings.ode.optionspad, settings);
    
    z = -Ypad(end, 3);
    [~, a, ~, ~] = atmosisa(settings.z0 + z); 
    
    %% retrive the correct wind data
    if settings.wind.model
        Hour = settings.stoch.Hour; 
        Day = settings.stoch.Day; 
        t = Tpad(end); 

        if settings.stoch.N > 1
            [uw, vw, ww] = windMatlabGenerator(settings, z, t, Hour, Day);
        else
            [uw, vw, ww] = windMatlabGenerator(settings, z, t);
        end
    
    elseif settings.wind.input
        [uw, vw, ww] = windInputGenerator(settings, z, uncert);
    elseif  settings.wind.variable
    
        [uw, vw, ww] = windVariableGenerator(z, settings.stoch);
    else 
        uw = settings.stoch.uw; vw = settings.stoch.vw; ww = settings.stoch.ww;
    end
    inertialWind = [uw, vw, ww]; 
    
    %% compute the exit conditions in the aerodynamic frame 
    vExit = norm(Ypad(end, 4)); 
    machExit = vExit/a;
    
    bodyWind = quatrotate(Y0(10:13)', inertialWind);
    bodyVelocity = [Ypad(end, 4), 0, 0];
    Vr = bodyVelocity - bodyWind;
    ur = Vr(1); vr = Vr(2); wr = Vr(3);
    
    if ur > 1e-9
        alpha = atan(wr/ur)*180/pi;
        beta = atan(vr/ur)*180/pi;
    else
        alpha = 0; 
        beta = 0; 
    end
    
    %% create dissileMatcom input data struct
    dissileVars.Alpha = sort([-alpha, alpha, 0, -1, 1]); 
    dissileVars.Beta = beta; 
    dissileVars.Alt = settings.z0 + z; 
    dissileVars.Mach = machExit;
    dissileVars.xcg = interpLinear(settings.motor.expTime, settings.xcg, Tpad(end));
    dissileVars.hprot = []; 
    input = createDissileInput(settings, dissileVars);
    
    input.flagXCP = true;
    coeff = dissileMatcom(input); 

    vars.useAlphaTot = false; 
    vars.checkOpenRocket = false; 
    vars.modHaack = true; 
    
    if dissileVars.Alpha(1) == alpha
        iAlpha = 1; 
    else
        iAlpha = 5; 
    end
    
    %% compute the stability margin and coresponding quality indices 
    [XCPvect, qualityIndex, stabilityCoeffs] = getXCP(coeff, iAlpha, 1, input, vars);
    data_stability.time = Tpad; 
    data_stability.state = Ypad; 
    data_stability.wind = [uw, vw, ww]; 
    data_stability.XCPvect = XCPvect; 
    data_stability.qualityIndex = qualityIndex; 
    data_stability.stabilityCoeffs = stabilityCoeffs; 

end 


